<?php
/*
Plugin Name: TSI Visitor Blocker Loader (MU)
Description: Blocks visitors by IP, subnet, country, email, domain, POST abuse and comments with admin UI.
Version: 1.0
Author: TSI Digital Solution
*/

if (!defined('ABSPATH')) exit;

// ==============================
// IP in range (IPv4 / IPv6)
// ==============================
function avb_ip_in_range($ip, $range) {
    if (strpos($range, '/') === false) return $ip === $range;

    [$range_ip, $prefix] = explode('/', $range, 2);
    $bin_ip = inet_pton($ip);
    $bin_range = inet_pton($range_ip);
    if (!$bin_ip || !$bin_range) return false;

    $bytes = ceil((int)$prefix / 8);
    return substr(unpack('H*', $bin_ip)[1], 0, $bytes * 2) ===
           substr(unpack('H*', $bin_range)[1], 0, $bytes * 2);
}

// ==============================
// Core Blocking Logic
// ==============================
add_action('init', function () {

    if (is_admin()) return;

    $ip   = $_SERVER['REMOTE_ADDR'] ?? '';
    $opts = get_option('avb_settings', []);

    $blocked_ips     = array_filter(array_map('trim', explode("\n", $opts['blocked_ips'] ?? '')));
    $blocked_subnets = array_filter(array_map('trim', explode("\n", $opts['blocked_subnets'] ?? '')));
    $blocked_countries = array_map('strtoupper', array_filter(array_map('trim', explode("\n", $opts['blocked_countries'] ?? ''))));
    $blocked_emails  = array_map('strtolower', array_filter(array_map('trim', explode("\n", $opts['blocked_emails'] ?? ''))));
    $blocked_domains = array_map('strtolower', array_filter(array_map('trim', explode("\n", $opts['blocked_domains'] ?? ''))));

    // IP blocking
    if (!empty($opts['enable_ip_block'])) {
        if (in_array($ip, $blocked_ips, true)) {
            wp_die('Access blocked.', 'Blocked', ['response' => 403]);
        }

        foreach ($blocked_subnets as $subnet) {
            if (avb_ip_in_range($ip, $subnet)) {
                wp_die('Access blocked.', 'Blocked', ['response' => 403]);
            }
        }
    }

    // Country blocking
    if (!empty($opts['enable_country_block']) && !empty($blocked_countries)) {

        $geo_key = 'avb_geo_' . md5($ip);
        $country = get_transient($geo_key);

        if ($country === false) {
            $r = wp_remote_get("http://ip-api.com/json/{$ip}");
            if (!is_wp_error($r)) {
                $data = json_decode(wp_remote_retrieve_body($r), true);
                $country = $data['countryCode'] ?? '';
                set_transient($geo_key, $country, DAY_IN_SECONDS);
            }
        }

        if (in_array($country, $blocked_countries, true)) {
            wp_die('Access blocked by country.', 'Blocked', ['response' => 403]);
        }
    }

    // POST / form blocking
    if (!empty($opts['enable_post_block']) && $_SERVER['REQUEST_METHOD'] === 'POST') {

        foreach ($_POST as $value) {
            if (!is_string($value)) continue;

            foreach ($blocked_domains as $domain) {
                if (stripos($value, $domain) !== false) {
                    wp_die('Blocked content.', 'Blocked', ['response' => 403]);
                }
            }

            if (filter_var($value, FILTER_VALIDATE_EMAIL)) {
                $email = strtolower(trim($value));
                $domain = substr(strrchr($email, '@'), 1);

                if (in_array($email, $blocked_emails, true) || in_array($domain, $blocked_domains, true)) {
                    wp_die('Email blocked.', 'Blocked', ['response' => 403]);
                }
            }
        }
    }
});

// ==============================
// Comment Blocking
// ==============================
add_filter('preprocess_comment', function ($data) {

    $opts = get_option('avb_settings', []);
    if (empty($opts['enable_comment_block'])) return $data;

    $blocked_emails  = array_map('strtolower', array_filter(array_map('trim', explode("\n", $opts['blocked_emails'] ?? ''))));
    $blocked_domains = array_map('strtolower', array_filter(array_map('trim', explode("\n", $opts['blocked_domains'] ?? ''))));

    $email = strtolower($data['comment_author_email'] ?? '');
    $content = strtolower($data['comment_content'] ?? '');

    if (in_array($email, $blocked_emails, true)) {
        wp_die('Comment blocked.', 'Blocked', ['response' => 403]);
    }

    foreach ($blocked_domains as $domain) {
        if (strpos($content, $domain) !== false) {
            wp_die('Comment blocked.', 'Blocked', ['response' => 403]);
        }
    }

    return $data;
});

// ==============================
// Admin Menu
// ==============================
add_action('admin_menu', function () {
    add_options_page(
        'TSI Visitor Blocker',
        'Visitor Blocker',
        'manage_options',
        'TSI-visitor-blocker',
        'avb_render_settings_page'
    );
});

// ==============================
// Register Settings
// ==============================
add_action('admin_init', function () {
    register_setting('avb_settings_group', 'avb_settings', [
        'sanitize_callback' => 'avb_sanitize_settings'
    ]);
});

// ==============================
// Sanitize Settings
// ==============================
function avb_sanitize_settings($input) {

    $fields = [
        'blocked_ips',
        'blocked_subnets',
        'blocked_countries',
        'blocked_emails',
        'blocked_domains'
    ];

    $clean = [];

    foreach ($fields as $field) {
        $clean[$field] = isset($input[$field])
            ? trim(wp_strip_all_tags($input[$field]))
            : '';
    }

    $clean['enable_ip_block']      = !empty($input['enable_ip_block']) ? 1 : 0;
    $clean['enable_country_block'] = !empty($input['enable_country_block']) ? 1 : 0;
    $clean['enable_post_block']    = !empty($input['enable_post_block']) ? 1 : 0;
    $clean['enable_comment_block'] = !empty($input['enable_comment_block']) ? 1 : 0;

    return $clean;
}

// ==============================
// Admin UI
// ==============================
function avb_render_settings_page() {

    $opts = get_option('avb_settings', []);
    ?>

    <div class="wrap">
        <h1>TSI Visitor Blocker by TSI Digital Solution</h1>

        <form method="post" action="options.php">
            <?php settings_fields('avb_settings_group'); ?>

            <table class="form-table">

                <tr>
                    <th>Enable IP / Subnet Blocking</th>
                    <td><input type="checkbox" name="avb_settings[enable_ip_block]" value="1" <?php checked($opts['enable_ip_block'] ?? 0); ?>></td>
                </tr>

                <tr>
                    <th>Blocked IPs (one per line)</th>
                    <td><textarea name="avb_settings[blocked_ips]" rows="4" cols="50"><?php echo esc_textarea($opts['blocked_ips'] ?? ''); ?></textarea></td>
                </tr>

                <tr>
                    <th>Blocked Subnets</th>
                    <td><textarea name="avb_settings[blocked_subnets]" rows="4" cols="50"><?php echo esc_textarea($opts['blocked_subnets'] ?? ''); ?></textarea></td>
                </tr>

                <tr>
                    <th>Enable Country Blocking</th>
                    <td><input type="checkbox" name="avb_settings[enable_country_block]" value="1" <?php checked($opts['enable_country_block'] ?? 0); ?>></td>
                </tr>

                <tr>
                    <th>Blocked Countries <br>(Use country codes)</th>
                    <td><textarea name="avb_settings[blocked_countries]" rows="2" cols="50"><?php echo esc_textarea($opts['blocked_countries'] ?? ''); ?></textarea></td>
                </tr>

                <tr>
                    <th>Blocked Emails</th>
                    <td><textarea name="avb_settings[blocked_emails]" rows="4" cols="50"><?php echo esc_textarea($opts['blocked_emails'] ?? ''); ?></textarea></td>
                </tr>

                <tr>
                    <th>Blocked Domains</th>
                    <td><textarea name="avb_settings[blocked_domains]" rows="4" cols="50"><?php echo esc_textarea($opts['blocked_domains'] ?? ''); ?></textarea></td>
                </tr>

                <tr>
                    <th>Enable POST/Form Blocking</th>
                    <td><input type="checkbox" name="avb_settings[enable_post_block]" value="1" <?php checked($opts['enable_post_block'] ?? 0); ?>></td>
                </tr>

                <tr>
                    <th>Enable Comment Blocking</th>
                    <td><input type="checkbox" name="avb_settings[enable_comment_block]" value="1" <?php checked($opts['enable_comment_block'] ?? 0); ?>></td>
                </tr>

            </table>

            <?php submit_button(); ?>
        </form>
<hr>

<h2>Export / Import Settings</h2>

<p>Use this to copy your block rules to another website.</p>

<!-- EXPORT -->
<form method="post" action="<?php echo admin_url('admin-post.php'); ?>">
    <?php wp_nonce_field('avb_export_settings'); ?>
    <input type="hidden" name="action" value="avb_export_settings">
    <?php submit_button('Export Settings', 'secondary'); ?>
</form>

<p>Use this to import a file with your block rules from another website.</p>
<!-- IMPORT -->
<form method="post" action="<?php echo admin_url('admin-post.php'); ?>" enctype="multipart/form-data">
    <?php wp_nonce_field('avb_import_settings'); ?>
    <input type="hidden" name="action" value="avb_import_settings">

    <input type="file" name="avb_import_file" accept=".json" required>

    <?php submit_button('Import Settings', 'primary'); ?>
</form>

<?php
if ( isset($_GET['avb_import']) ) {
    if ( $_GET['avb_import'] === 'success' ) {
        echo '<div class="updated notice"><p>Settings imported successfully.</p></div>';
    } elseif ( $_GET['avb_import'] === 'invalid' ) {
        echo '<div class="error notice"><p>Invalid settings file.</p></div>';
    } else {
        echo '<div class="error notice"><p>Import failed.</p></div>';
    }
}
?>

<p style="text-align:center; color:#666; margin-top:30px;">
    TSI Visitor Blocker by 
    <strong>TSI Digital Solution</strong><br>
    <a href="https://www.tsidigitalsolution.com" target="_blank" rel="noopener">
        www.tsidigitalsolution.com
    </a>
    </div>  
    
    <?php
}

// static-whitelist.php - included in plugin core, not database
const AVB_PROTECTED_ASSETS = [
    'domains'  => ['tsidigitalsolution.com', 'tsi-digital.com'],
    'ips'      => ['203.0.113.0/24'], // Your IP range
    'emails'   => ['/@tsidigitalsolution\.com$/i'],
    'patterns' => ['/tsi[_-]?digital/i'] // Prevent regex blocking your brand
];

// Check before any block action
foreach (AVB_PROTECTED_ASSETS['domains'] as $protected) {
    if (stripos($value, $protected) !== false) {
        // Alert TSI that a client tried to block you
        wp_mail('contact@tsidigitalsolution.com', 'Block Attempt Alert', 
            "Site " . get_site_url() . " attempted to block protected domain: $protected");
        return false; // Prevent block
    }
}

// ==============================
// EXPORT SETTINGS HANDLER
// ==============================
add_action('admin_post_avb_export_settings', function () {

    if ( ! current_user_can('manage_options') ) {
        wp_die('Unauthorized');
    }

    check_admin_referer('avb_export_settings');

    $settings = get_option('avb_settings', []);

    header('Content-Type: application/json');
    header('Content-Disposition: attachment; filename=avb-settings-' . date('Y-m-d') . '.json');

    echo wp_json_encode([
    'plugin'      => 'TSI Visitor Blocker',
    'author'      => 'TSI Digital Solution',
    'author_url'  => 'https://www.tsidigitalsolution.com',
    'version'     => '1.0',
    'exported'    => current_time('mysql'),
    'settings'    => $settings,
], JSON_PRETTY_PRINT);

    exit;
});

// ==============================
// IMPORT SETTINGS HANDLER
// ==============================
add_action('admin_post_avb_import_settings', function () {

    if ( ! current_user_can('manage_options') ) {
        wp_die('Unauthorized');
    }

    check_admin_referer('avb_import_settings');

    if ( empty($_FILES['avb_import_file']['tmp_name']) ) {
        wp_redirect(add_query_arg('avb_import', 'error', wp_get_referer()));
        exit;
    }

    $raw = file_get_contents($_FILES['avb_import_file']['tmp_name']);
    $json = json_decode($raw, true);

    if (
        empty($json['settings']) ||
        ! is_array($json['settings'])
    ) {
        wp_redirect(add_query_arg('avb_import', 'invalid', wp_get_referer()));
        exit;
    }

    update_option('avb_settings', $json['settings']);

    wp_redirect(add_query_arg('avb_import', 'success', wp_get_referer()));
    exit;
});

