<?php
/**
 * Advanced Visitor Blocker – Helpers
 * Used by core.php
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * =====================================
 * Get real visitor IP (Cloudflare safe)
 * =====================================
 */
function avb_get_visitor_ip() {

    $sources = [
        'HTTP_CF_CONNECTING_IP', // Cloudflare
        'HTTP_X_FORWARDED_FOR',
        'HTTP_X_REAL_IP',
        'REMOTE_ADDR',
    ];

    foreach ( $sources as $source ) {
        if ( ! empty( $_SERVER[ $source ] ) ) {
            $ips = explode( ',', $_SERVER[ $source ] );
            $ip  = trim( $ips[0] );

            if ( filter_var( $ip, FILTER_VALIDATE_IP ) ) {
                return $ip;
            }
        }
    }

    return '';
}

/**
 * =====================================
 * Normalize textarea lists
 * =====================================
 */
function avb_parse_list( $value, $case = 'lower' ) {

    if ( empty( $value ) ) {
        return [];
    }

    $lines = preg_split( '/\r\n|\r|\n/', $value );
    $clean = [];

    foreach ( $lines as $line ) {
        $line = trim( $line );
        if ( $line === '' ) {
            continue;
        }

        if ( $case === 'lower' ) {
            $line = strtolower( $line );
        } elseif ( $case === 'upper' ) {
            $line = strtoupper( $line );
        }

        $clean[] = $line;
    }

    return array_unique( $clean );
}

/**
 * =====================================
 * Extract domain from email
 * =====================================
 */
function avb_get_email_domain( $email ) {
    return strtolower( substr( strrchr( $email, '@' ), 1 ) );
}

/**
 * =====================================
 * Country lookup (cached)
 * =====================================
 */
function avb_get_country_code( $ip ) {

    if ( empty( $ip ) ) {
        return '';
    }

    $cache_key = 'avb_geo_' . md5( $ip );
    $country   = get_transient( $cache_key );

    if ( $country !== false ) {
        return $country;
    }

    $response = wp_remote_get(
        "http://ip-api.com/json/{$ip}",
        [ 'timeout' => 5 ]
    );

    if ( is_wp_error( $response ) ) {
        return '';
    }

    $data = json_decode( wp_remote_retrieve_body( $response ), true );
    $country = strtoupper( $data['countryCode'] ?? '' );

    set_transient( $cache_key, $country, DAY_IN_SECONDS );

    return $country;
}

/**
 * =====================================
 * Logging (optional)
 * =====================================
 */
function avb_log_event( $type, $message, $ip = '' ) {

    $logs = get_option( 'avb_logs', [] );

    $logs[] = [
        'time'    => current_time( 'mysql' ),
        'ip'      => $ip ?: avb_get_visitor_ip(),
        'type'    => sanitize_text_field( $type ),
        'message' => sanitize_text_field( $message ),
    ];

    if ( count( $logs ) > 300 ) {
        $logs = array_slice( $logs, -300 );
    }

    update_option( 'avb_logs', $logs );
}

/**
 * =====================================
 * Centralized block handler
 * =====================================
 */
function avb_block( $reason = 'Access blocked.' ) {

    if ( is_admin() ) {
        return;
    }

    status_header( 403 );
    wp_die(
        esc_html( $reason ),
        esc_html__( 'Blocked', 'advanced-visitor-blocker' ),
        [ 'response' => 403 ]
    );
}
